function Fit = SubtrZLP(hObject,handles)
% For the deconvolution to work propperly, the ZLP has to be subtracted
% from the data. In case no vacuum ZLP was recorded, this can be done by fitting a Gaussian, a Weibull or a
% Nordheim-Fowler to the ZLP. In case a vacuum ZLP was recorded, it can
% simply be denoised and fitted to the data. In case a ZLP was recorded
% before and after the measurement, a mixed ZLP can be fitted to the data
% allowing to account for changes over time

global tracer

if isempty(tracer)
    tracer=1;
else
    Fit    = [];
    clearvars -except Fit
    return
end

%% Load File
OrgImg     = load(fullfile(tempdir,'ShowOrg.mat')).image;
image      = OrgImg;
Add        = load(fullfile(tempdir,'Add.mat')).Add;
FitOld     = load(fullfile(tempdir,'SubtrZLP.mat')).Data;
FitOld     = FitOld{4};

%% Get Handles
Energies  = get(handles.ZLPAlign,'UserData');
FWHM      = get(handles.ShowFWHM,'UserData');
ZLP_load  = get(handles.LoadPeak,'UserData');
Method    = get(handles.SubMeth,'Value');
Name      = get(handles.SubMeth,'String');
Name      = Name{Method};
EnergyRes = get(handles.Slider, 'SliderStep');
EnergyRes = EnergyRes(1);
FitDispOpt= get(handles.FitDispOpt,'Value');
DiffMult  = str2double(get(handles.DiffMult,'String'));
Width     = str2double(get(handles.Width,'String'));

if Method==4 && ~any(Name=='1')
    Method=5;
end

if ~isempty(Add)
    SizeAdd = size(Add,3);
else
    SizeAdd = 0;
end

Sum        = sum(OrgImg,3);
SumDen     = sum(image,3);
clearvars OrgImg
StartPar   = get(handles.DiffMult,'UserData');
EWidth     = StartPar{2,2};
EStart     = StartPar{2,1};

Energies  = [Energies,linspace(Energies(1,end) + EnergyRes, Energies(1,end) + EnergyRes*(SizeAdd-1),SizeAdd)];
OrgLength = size(image,3);
image     = cat(3,image,Add);

%% Subtract ZLP
Spec       = image./sum(image,3);
FWHMOrg    = FWHM;

switch FitDispOpt
    case 1
        Spec  = reshape(Spec,[size(image,1).*size(image,2),size(image,3)]);
        Spec  = mean(Spec,1);
        Spec  = Spec/max(Spec,[],2);
        FWHM  = mean(FWHM,'all');
        Sigma = FWHM./(2*sqrt(2.*log(2)));
    case 2
        Spec  = mean(Spec,2);
        Spec  = reshape(Spec,[size(image,1),size(image,3)]);
        Spec  = Spec./max(Spec,[],2);
        FWHM  = mean(FWHM,2);
        Sigma = FWHM./(2*sqrt(2.*log(2)));
    case 3
        Spec  = reshape(Spec,[size(image,1).*size(image,2),size(image,3)]);
        Spec  = Spec./max(Spec,[],2);
        FWHM  = reshape(FWHM,[size(image,1).*size(image,2),1]);
        Sigma = FWHM./(2*sqrt(2.*log(2)));
end

if isempty(EStart)
    EStart= -Width/4*mean(FWHM,'all');
    EWidth=  Width/2*mean(FWHM,'all');
else
    EStart =  max(EStart,-mean(FWHM,'all')*Width/2);
    if EStart+EWidth>mean(FWHM,'all')*Width/2
        EWidth =  mean(FWHM,'all')*Width/2;
    end
end

index1     = find(Energies<= EStart,1,'last');
index2     = find(Energies>= EStart+EWidth,1,'first');
E          = Energies(index1:index2);
Peak       = Spec(:,index1:index2);
options    = optimset('Display','off');
Fit        = zeros(size(Spec));


if ~isempty(ZLP_load{1,1})
    ZLP_load1  = ZLP_load{1,1};
    ZLP_load1  = repmat(ZLP_load1,[size(Spec,1),1]);
end

if ~isempty(ZLP_load{2,1})
    ZLP_load2  = ZLP_load{2,1};
    ZLPVAR2    = ZLP_load{2,2};
    ZLP_load2  = repmat(ZLP_load2,[size(Spec,1),1]);
end

switch Method
    case 1
        %% Fowler-Nordheim
        for i=1:size(Spec,1)
            x           = [ 1 , 0 , 1, 0.5 ];
            eqn         = @(x) sum((x(1).*exp(-(E-real(x(2)))./x(3))./(1+exp(-(E-real(x(2)))./x(4))) - Peak(i,:)).^2);
            x           = fminsearch(eqn,x,options);
            Fit(i,:)    = x(1).*exp(-(Energies-real(x(2)))./x(3))./(1+exp(-(Energies-real(x(2)))./x(4)));
        end
    case 2
        %% Gauss
        for i=1:size(Spec,1)
            x           = [ 1 , 0 , Sigma(i)];
            eqn         = @(x) sum((x(1).*exp(-((E-x(2))./(sqrt(2).*x(3))).^2)- Peak(i,:)).^2);
            x           = fminsearch(eqn,x,options);
            Fit(i,:)     = x(1).*exp(-((Energies-x(2))./(sqrt(2).*x(3))).^2);
        end
    case 3
        %% Weibull
        for i=1:size(Spec,1)
            x           = [0.00044883, 1.7517 , -1.8317 ,14.204 ,  3.2323e-06];
            eqn         = @(x) sum(( x(1) .* ((E-x(2))./x(3)).^(x(4)-1).* exp(((E-x(2))./x(3)).^x(4))+x(5) - Peak(i,:)).^2);
            x           = fminsearch(eqn,x,options);
            Fit(i,:)    = x(1) .* ((Energies-x(2))./x(3)).^(x(4)-1).* exp(((Energies-x(2))./x(3)).^x(4));
        end
    case 4
        %% Loaded ZLP1
        for i=1:size(Spec,1)
            ZLPfit1     = ZLP_load1(i,index1:index2);
            x           =  1.1;
            eqn         = @(x) sum((abs(x(1)).*ZLPfit1 - Peak(i,:)).^2);
            x           = fminsearch(eqn,x,options);
            Fit(i,:)    = abs(x(1)).*ZLP_load1(i,:);
        end
    case 5
        %% Loaded ZLP2
        for i=1:size(Spec,1)
            ZLPfit2     = ZLP_load2(i,index1:index2);
            x           =  1.1;
            eqn         = @(x) sum((abs(x(1)).*ZLPfit2 - Peak(i,:)).^2);
            x           = fminsearch(eqn,x,options);
            Fit(i,:)    = abs(x(1)).*ZLP_load2(i,:);
        end
    case 6
        %% Dual ZLP
        for i=1:size(Spec,1)
            ZLPfit1     = ZLP_load1(i,index1:index2);
            ZLPfit2     = ZLP_load2(i,index1:index2);
            x           = [0.5 , 0.5];
            eqn         = @(x) sum((abs(x(1)).*ZLPfit1+ abs(x(2)).*ZLPfit2-Peak(i,:)).^2);
            x           = fminsearch(eqn,x,options);
            Fit(i,:)    = abs(x(1)).*ZLP_load1(i,:) + abs(x(2)).*ZLP_load2(i,:);
        end
end

fit       = Fit./sum(Spec,2);
clearvars Fit
spec      = Spec./sum(Spec,2);
res       = (spec-fit)*DiffMult;
ResPos    = res;
ResPos(ResPos<0)=0;
ResNeg    = res;
ResNeg(ResPos>0)=0;
ResPos    = mean(ResPos,1);
ResNeg    = mean(ResNeg,1);
spec      = mean(spec,1);
fit       = mean(fit,1);

fig       = figure('Name','Select ZLP fit region - double click to confirm!');
plot(Energies,spec,'Color','b');
hold on
plot(Energies,fit,'Color','m');
plot(Energies,ResPos,'Color','g');
plot(Energies,ResNeg,'Color','r');
plot(Energies,zeros(size(Energies)),'--','Color','k');
legend('Normed mean Spectrum','Fit',['Pos. Difference x',num2str(DiffMult)],['Neg. Difference x',num2str(DiffMult)]);
ylabel('mean Loss probability [a.u.]');
xlabel('Loss energy [eV]');

hold off

xlim([-mean(FWHM,'all')*Width/2,mean(FWHM,'all')*Width/2]);
ylim([-max(spec)/2,max(spec).*1.25])

try
    Select  = drawrectangle('Position',[EStart,-max(spec)/2,EWidth,max(spec)/2+max(spec).*1.25],'Color','b','StripeColor','w');
    l1      = addlistener(Select,'MovingROI',@(src,evt) LiveFitZLP(src,evt,fig,handles,image./sum(image,3),Energies,FWHMOrg));
    l2      = addlistener(Select,'ROIClicked',@(src,evt) DoubleClicker(src,evt));
    set(handles.LiveFitPanel,'UserData',{2,Select,fig,image./sum(image,3),Energies,FWHMOrg});
    guidata(hObject,handles)
    uiwait;
    set(handles.LiveFitPanel,'UserData',[]);
    guidata(hObject,handles)
    pos     = Select.Position;
    EStart  = pos(1);
    EWidth  = pos(3);
    Ind1    = pos(1);
    Ind2    = pos(1)+pos(3);
    delete(Select);
    close(fig);
    delete(l1);
    delete(l2);
catch
    Fit=[];
    set(handles.LiveFitPanel,'UserData',[]);
    guidata(hObject,handles)
    clearvars -except Fit
    clear global
    return
end
Ind1     = find(Energies<=Ind1,1,'last');
Ind2     = find(Energies>=Ind2,1,'first');
options  = optimset('Display','off');

%% Update Method
guidata(hObject, handles);
Method     = get(handles.SubMeth,'Value');
Name       = get(handles.SubMeth,'String');
Name       = Name{Method};
ZLP_load   = get(handles.LoadPeak,'UserData');
FWHM       = get(handles.ShowFWHM,'UserData');

if Method==4 && ~any(Name=='1')
    Method=5;
end

if ~isempty(ZLP_load{1,1})
    ZLP_load1  = ZLP_load{1,1};
    ZLPVAR1    = ZLP_load{1,2};
    ZLPVAR1    = reshape(ZLPVAR1,[1,1,size(ZLP_load1,2)]);
    ZLPVAR1    = repmat(ZLPVAR1,[size(FWHM,1),size(FWHM,2),1]);
    ZLP_load1  = reshape(ZLP_load1,[1,1,size(ZLP_load1,2)]);
    ZLP_load1  = repmat(ZLP_load1,[size(FWHM,1),size(FWHM,2),1]);
end

if ~isempty(ZLP_load{2,1})
    ZLP_load2  = ZLP_load{2,1};
    ZLPVAR2    = ZLP_load{2,2};   
    ZLPVAR2    = reshape(ZLPVAR2,[1,1,size(ZLP_load2,2)]);
    ZLPVAR2    = repmat(ZLPVAR2,[size(FWHM,1),size(FWHM,2),1]);        
    ZLP_load2  = reshape(ZLP_load2,[1,1,size(ZLP_load2,2)]);
    ZLP_load2  = repmat(ZLP_load2,[size(FWHM,1),size(FWHM,2),1]);
end

StartPar{2,2}  = EWidth;
StartPar{2,1}  = EStart;
set(handles.DiffMult,'UserData',StartPar);
ZLPVAR   = zeros(size(image));
Max      = max(image,[],3);
image    = image./Max;

%% Fit
Fit       = zeros(size(image));
WaitTics  = size(image,1);
Sigma     = FWHMOrg./(2*sqrt(2.*log(2)));

WaitZLP   = parfor_wait(WaitTics, 'Waitbar', true,'ReportInterval',1);
for i=1:size(image,1)
    for j=1:size(image,2)
        Img         = reshape(image(i,j,:),[1,size(image,3)]);
        Peak        = Img(Ind1:Ind2);
        E           = Energies(Ind1:Ind2);
        switch Method
            case 1
                %% Fowler-Nordheim
                x           = [ 1 ,0 , 0.2, 0.0825];
                eqn         = @(x) sum((x(1).*exp(-(E-x(2))./abs(x(3)))./(1+exp(-(E-x(2))./abs(x(4))))-Peak).^2);
                x           = fminsearch(eqn,x,options);
                fit         = x(1).*exp(-(Energies-x(2))./abs(x(3)))./(1+exp(-(Energies-x(2))./abs(x(4))));
            case 2
                %% Gauss
                x           = [ 1 , 0 , Sigma(i,j), 10^(-5)];
                eqn         = @(x) sum((x(1).*exp(-((E-x(2))./(sqrt(2).*x(3))).^2)+x(4)-Peak).^2);
                x           = fminsearch(eqn,x,options);
                fit         = x(1).*exp(-((Energies-x(2))./(sqrt(2).*x(3))).^2);
            case 3
                %% Weibull
                x           =  [0.00044883, 1.7517 , -1.8317 ,14.204 ,  3.2323e-06];
                eqn         = @(x) sum(( x(1) .* ((E-x(2))./x(3)).^(x(4)-1).* exp(((E-x(2))./x(3)).^x(4))+x(5) -Peak).^2);
                x           = fminsearch(eqn,x,options);
                fit         =  x(1) .* ((Energies-x(2))./x(3)).^(x(4)-1).* exp(((Energies-x(2))./x(3)).^x(4));
            case 4
                %% Loaded ZLP1
                zlp_load1   = reshape(ZLP_load1(i,j,:),[1,size(ZLP_load1,3)]);
                zlpvar      = reshape(ZLPVAR1(i,j,:),[1,size(ZLP_load1,3)]);
                ZLPfit1     = zlp_load1(1,Ind1:Ind2);
                x           = 1.1;
                eqn         = @(x) sum((abs(x(1)).*ZLPfit1-Peak).^2);
                x           = fminsearch(eqn,x,options);
                fit         = abs(x(1)).*zlp_load1;
                zlpvar      = abs(x(1))^2.*zlpvar;
            case 5
                %% Loaded ZLP2
                zlp_load2   = reshape(ZLP_load2(i,j,:),[1,size(ZLP_load2,3)]);
                zlpvar      = reshape(ZLPVAR2(i,j,:),[1,size(ZLP_load2,3)]);
                ZLPfit2     = zlp_load2(1,Ind1:Ind2);
                x           = 1.1;
                eqn         = @(x) sum((abs(x(1)).*ZLPfit2-Peak).^2);
                x           = fminsearch(eqn,x,options);
                fit         = abs(x(1)).*zlp_load2;
                zlpvar      = abs(x(1))^2.*zlpvar;
            case 6
                %% Loaded Dual ZLP
                zlp_load1   = reshape(ZLP_load1(i,j,:),[1,size(ZLP_load1,3)]);
                zlp_load2   = reshape(ZLP_load2(i,j,:),[1,size(ZLP_load2,3)]);
                zlpvar1     = reshape(ZLPVAR1(i,j,:),[1,size(ZLP_load1,3)]);
                zlpvar2     = reshape(ZLPVAR2(i,j,:),[1,size(ZLP_load2,3)]);
                ZLPfit1     = zlp_load1(1,Ind1:Ind2);
                ZLPfit2     = zlp_load2(1,Ind1:Ind2);
                x           = [0.5 , 0.5];
                eqn         = @(x) sum((abs(x(1)).*ZLPfit1+ abs(x(2)).*ZLPfit2 -Peak).^2);
                x           = fminsearch(eqn,x,options);
                fit         = abs(x(1)).*zlp_load1 + abs(x(2)).*zlp_load2;
                zlpvar      = abs(x(1))^2.*zlpvar1 + abs(x(2))^2.*zlpvar2;
        end
        
        %%
        image(i,j,:)  = Img-fit;
        Fit(i,j,:)    = fit;
        ZLPVAR(i,j,:) = zlpvar;
    end
    %% Check Waitbar exist
    Flag = CheckWaitbar1(WaitZLP);
    if Flag==1
        Fit=[];
        clearvars -except Fit
        clear global
        return
    end
    WaitZLP.Send;
end
WaitZLP.Destroy;

image  = image.*Max;
Fit    = Fit.*Max;
ZLPVAR = ZLPVAR.*Max.^2;

%% Finish
% as the subtraction of the ZLP often leads to huge leftovers, negative and
% positive in the direct ZLP region, a cutoff can be set manually to separate it
% from the signal

PosSpec    = image./sum(image,3);
PosSpec(PosSpec<0) = 0;
NegSpec    = image./sum(image,3);
NegSpec(NegSpec>0) = 0;
PosSpec    = reshape(PosSpec(:,:,1:OrgLength),[size(image,1)*size(image,2),OrgLength]);
PosSpec    = mean(PosSpec,1);
NegSpec    = reshape(NegSpec(:,:,1:OrgLength),[size(image,1)*size(image,2),OrgLength]);
NegSpec    = mean(NegSpec,1);
Energies   = Energies(1,1:OrgLength);

fig        = figure('Name','Select ZLP residuals to delete - double click to confirm!');
plot(Energies,PosSpec,'g',Energies,NegSpec,'r');
hold on
plot(Energies,zeros(size(Energies)),'--','Color','k');
hold off
xlim([min(Energies),mean(FWHM,'all')*5]);
ylim([min(NegSpec),max(PosSpec).*1.25])
ylabel('mean Loss probability [a.u.]');
xlabel('Loss energy [eV]');
legend('Pos. sum spectra','Neg. sum spectra')

try
    Select  = drawrectangle('Position',[max(-mean(FWHM,'all')*3,Energies(1,1)),min(NegSpec),mean(FWHM,'all')*3,-min(NegSpec) + max(PosSpec).*1.25],'Color','b','StripeColor','w');
    l       = addlistener(Select,'ROIClicked',@(src,evt) DoubleClicker(src,evt));
    uiwait;
    pos       = Select.Position;
    Index1    = pos(1);
    Index2    = pos(1)+pos(3);
    Index1    = find(Energies>=Index1,1,'first');
    Index2    = find(Energies<=Index2,1,'last');
    delete(Select);
    close(fig);
    delete(l);
    cutoff=[Index1,Index2];
catch
    cutoff = [];
end

if ~isempty(FitOld)
    fig=figure('Name','Comparisson to old Fit');
    plot(Energies,PosSpec,'g',Energies,NegSpec,'r');
    hold on
    plot(Energies,FitOld(1,:),'b',Energies,FitOld(2,:),'m');
    plot(Energies,zeros(size(Energies)),'--','Color','k');
    hold off
    xlim([min(Energies),mean(FWHM,'all')*5]);
    ylim([min(NegSpec),max(PosSpec).*1.25])
    ylabel('mean Loss probability [a.u.]');
    xlabel('Loss energy [eV]');
    legend('Pos. sum spectra','Neg. sum spectra',' old Pos. sum spectra','old Neg. sum spectra')
    fig.UserData.OnCleanup = onCleanup(@()CleanUp);                       % tracks deleted Object
    uiwait;
end

answer = questdlg('Use the subtraction?','Save result','Yes','No','No');
switch answer
    case 'Yes'
        Fit    = Fit./repmat(SumDen,[1,1,size(image,3)]);
        Fit    = Fit.*repmat(Sum,[1,1,size(image,3)]);
        ZLPVAR = ZLPVAR.*(repmat(Sum,[1,1,size(image,3)])./repmat(SumDen,[1,1,size(image,3)])).^2;
        Fit    = {Fit,cutoff,Method,[PosSpec;NegSpec],ZLPVAR};
    otherwise
        Fit         = [];
        clearvars -except Fit
        clear global
        return
end

%% Create Logbook

SubtrZLPLog = {['Method: ', Name, '  |  StartVal: ',num2str(Energies(Ind1)),'eV |  EndVal: ',num2str(Energies(Ind2)),'eV']};


switch Method
    case 4
        ZLP1Log    = load(fullfile(tempdir,'AlignZLP1Log.mat')).AlignZLPLog;
        
        for i=1:size(ZLP1Log,1)
            ZLP1Log{i,:} = ['  ',ZLP1Log{i,:}];
        end
        SubtrZLPLog= [SubtrZLPLog{:,1}; ZLP1Log];  
        
    case 5
        ZLP2Log    = load(fullfile(tempdir,'AlignZLP2Log.mat')).AlignZLPLog;
        for i=1:size(ZLP2Log,1)
            ZLP2Log{i,:} = ['  ',ZLP2Log{i,:}];
        end
        SubtrZLPLog= [SubtrZLPLog{:,1}; ZLP2Log];
        
    case 6
        ZLP1Log    = load(fullfile(tempdir,'AlignZLP1Log.mat')).AlignZLPLog;
        for i=1:size(ZLP1Log,1)
            ZLP1Log{i,:} = ['  ',ZLP1Log{i,:}];
        end
        ZLP2Log    = load(fullfile(tempdir,'AlignZLP2Log.mat')).AlignZLPLog;
        for i=1:size(ZLP2Log,1)
            ZLP2Log{i,:} = ['  ',ZLP2Log{i,:}];
        end
        SubtrZLPLog= [SubtrZLPLog{:,1};' First ZLP:';ZLP1Log;[];' Second ZLP:';ZLP2Log];   
end
save(fullfile(tempdir,'SubtrZLPLog.mat'),'SubtrZLPLog');
clearvars -except Fit
clear global
end